// Dental Appointment Booking Website - Version 3 (HTML/CSS/JS)

// Services data
const services = [
  { 
    id: 'examination', 
    name: 'Осмотр', 
    icon: 'icons/service-examination.svg', 
    color: '#0EA5E9',
    description: 'Профилактический осмотр и консультация'
  },
  { 
    id: 'filling', 
    name: 'Пломба', 
    icon: 'icons/service-filling.svg', 
    color: '#06B6D4',
    description: 'Лечение кариеса и установка пломбы'
  },
  { 
    id: 'whitening', 
    name: 'Отбеливание', 
    icon: 'icons/service-whitening.svg', 
    color: '#22D3EE',
    description: 'Профессиональное отбеливание зубов'
  },
  { 
    id: 'implant', 
    name: 'Имплант', 
    icon: 'icons/service-implant.svg', 
    color: '#10B981',
    description: 'Установка зубного импланта'
  }
]

// Initialize page
function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'history'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initHistoryPage()
  }
  
  initSlider()
}

// Initialize history page
function initHistoryPage() {
  loadVisits()
  updateStats()
  
  // Tab switching
  const tabs = document.querySelectorAll('.tab')
  tabs.forEach(tab => {
    tab.addEventListener('click', function() {
      tabs.forEach(t => t.classList.remove('active'))
      this.classList.add('active')
      const tabType = this.dataset.tab
      renderVisits(tabType)
    })
  })
  
  // Initial render
  renderVisits('active')
  
  // Poll for storage changes
  setInterval(() => {
    loadVisits()
    updateStats()
  }, 500)
}

// Load visits from localStorage
function loadVisits() {
  const saved = localStorage.getItem('dentalVisits')
  let visits = { active: [], past: [] }
  
  if (saved) {
    try {
      visits = JSON.parse(saved)
    } catch (e) {
      visits = { active: [], past: [] }
    }
  } else {
    // Default visits
    visits = {
      active: [
        {
          id: 1,
          clinic: 'Стоматология "Улыбка"',
          service: 'Осмотр',
          date: '2025-03-28',
          status: 'active'
        },
        {
          id: 2,
          clinic: 'Клиника "ДентПро"',
          service: 'Пломба',
          date: '2025-03-30',
          status: 'active'
        }
      ],
      past: [
        {
          id: 3,
          clinic: 'Стоматология "Улыбка"',
          service: 'Отбеливание',
          date: '2025-03-15',
          painRating: 2,
          qualityRating: 5
        },
        {
          id: 4,
          clinic: 'Клиника "ДентПро"',
          service: 'Осмотр',
          date: '2025-03-10',
          painRating: 1,
          qualityRating: 5
        },
        {
          id: 5,
          clinic: 'Стоматология "Улыбка"',
          service: 'Пломба',
          date: '2025-03-05',
          painRating: 3,
          qualityRating: 4
        }
      ]
    }
    localStorage.setItem('dentalVisits', JSON.stringify(visits))
  }
  
  return visits
}

// Render visits
function renderVisits(type) {
  const visits = loadVisits()
  const container = document.getElementById('visitsContainer')
  if (!container) return
  
  const visitsList = type === 'active' ? visits.active : visits.past
  
  if (visitsList.length === 0) {
    container.innerHTML = `
      <div class="empty-state">
        <img src="icons/empty-visits.svg" alt="" class="empty-icon" width="72" height="72" />
        <p>${type === 'active' ? 'У вас нет активных визитов' : 'История визитов пуста'}</p>
      </div>
    `
    return
  }
  
  container.innerHTML = visitsList.map(visit => {
    const service = services.find(s => s.name === visit.service) || services[0]
    const iconPath = service.icon
    
    if (type === 'active') {
      return `
        <div class="visit-card">
          <div class="visit-header">
            <img src="${iconPath}" alt="${visit.service}" class="visit-icon" width="40" height="40" />
            <div class="visit-info">
              <h3 class="visit-clinic">${visit.clinic}</h3>
              <p class="visit-service">${visit.service}</p>
            </div>
          </div>
          <div class="visit-date">
            <img src="icons/detail-calendar.svg" alt="" class="detail-icon" width="20" height="20" />
            <span>${new Date(visit.date).toLocaleDateString('ru-RU', { 
              day: 'numeric', 
              month: 'long', 
              year: 'numeric' 
            })}</span>
          </div>
          <div class="visit-status active-badge">
            Запланирован
          </div>
        </div>
      `
    } else {
      const painLevel = visit.painRating ? getPainLevel(visit.painRating) : null
      const stars = visit.qualityRating ? renderStars(visit.qualityRating) : ''
      
      return `
        <div class="visit-card">
          <div class="visit-header">
            <img src="${iconPath}" alt="${visit.service}" class="visit-icon" width="40" height="40" />
            <div class="visit-info">
              <h3 class="visit-clinic">${visit.clinic}</h3>
              <p class="visit-service">${visit.service}</p>
            </div>
          </div>
          <div class="visit-date">
            <img src="icons/detail-calendar.svg" alt="" class="detail-icon" width="20" height="20" />
            <span>${new Date(visit.date).toLocaleDateString('ru-RU', { 
              day: 'numeric', 
              month: 'long', 
              year: 'numeric' 
            })}</span>
          </div>
          ${painLevel ? `
            <div class="visit-rating-section">
              <div class="rating-item">
                <span class="rating-label">Боль:</span>
                <div class="rating-value" style="color: ${painLevel.color}">
                  <img src="${painLevel.icon}" alt="" width="16" height="16" />
                  <span>${painLevel.text}</span>
                </div>
              </div>
            </div>
          ` : ''}
          ${visit.qualityRating ? `
            <div class="visit-rating-section">
              <div class="rating-item">
                <span class="rating-label">Качество:</span>
                <div class="stars-wrapper">
                  ${stars}
                </div>
              </div>
            </div>
          ` : ''}
        </div>
      `
    }
  }).join('')
}

function getPainLevel(rating) {
  if (rating <= 1) return { text: 'Минимальная', color: '#10B981', icon: 'icons/pain-low.svg' }
  if (rating <= 2) return { text: 'Легкая', color: '#22D3EE', icon: 'icons/pain-medium.svg' }
  if (rating <= 3) return { text: 'Умеренная', color: '#F59E0B', icon: 'icons/pain-high.svg' }
  return { text: 'Сильная', color: '#EF4444', icon: 'icons/pain-high.svg' }
}

function renderStars(rating) {
  return Array.from({ length: 5 }, (_, i) => {
    const filled = i < rating
    return `<img src="icons/star-${filled ? 'filled' : 'empty'}.svg" alt="" width="16" height="16" style="filter: ${filled ? 'none' : 'opacity(0.3)'}" />`
  }).join('')
}

// Update stats
function updateStats() {
  const visits = loadVisits()
  const stats = [
    { 
      value: visits.past.length, 
      label: 'Всего визитов',
      icon: 'icons/stat-calendar.svg',
      color: '#0EA5E9'
    },
    { 
      value: new Set(visits.past.map(v => v.clinic)).size, 
      label: 'Клиник посещено',
      icon: 'icons/stat-building.svg',
      color: '#06B6D4'
    },
    { 
      value: visits.past.length > 0 
        ? Math.round(visits.past.reduce((sum, v) => sum + (v.qualityRating || 0), 0) / visits.past.length * 10) / 10
        : 0, 
      label: 'Средняя оценка',
      icon: 'icons/stat-star.svg',
      color: '#22D3EE'
    }
  ]
  
  const statsGrid = document.getElementById('statsGrid')
  if (statsGrid) {
    statsGrid.innerHTML = stats.map(stat => `
      <div class="stat-card">
        <div class="stat-icon-wrapper" style="background: ${stat.color}15">
          <img src="${stat.icon}" alt="" class="stat-icon" width="32" height="32" />
        </div>
        <div class="stat-content">
          <div class="stat-value">${stat.value}</div>
          <div class="stat-label">${stat.label}</div>
        </div>
      </div>
    `).join('')
  }
  
  // Update tab counts
  const activeCount = document.getElementById('activeCount')
  const pastCount = document.getElementById('pastCount')
  if (activeCount) activeCount.textContent = visits.active.length
  if (pastCount) pastCount.textContent = visits.past.length
}

// Initialize book page
function initBookPage() {
  let selectedService = null
  
  const servicesGrid = document.getElementById('servicesGrid')
  const dateInput = document.getElementById('date')
  const clinicSelect = document.getElementById('clinicSelect')
  const submitBtn = document.getElementById('submitBtn')
  const bookingForm = document.getElementById('bookingForm')
  const successMessage = document.getElementById('successMessage')
  
  // Set date limits
  const today = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
  if (dateInput) {
    dateInput.min = today
    dateInput.max = maxDate
  }
  
  // Render services
  if (servicesGrid) {
    servicesGrid.innerHTML = services.map(service => `
      <button 
        type="button" 
        class="service-card" 
        data-service="${service.id}"
        style="border-color: rgba(14, 165, 233, 0.2); background: rgba(255, 255, 255, 0.9);"
      >
        <div class="service-icon-wrapper" style="background: ${service.color}15">
          <img src="${service.icon}" alt="${service.name}" class="service-icon" width="32" height="32" />
        </div>
        <div class="service-content">
          <h3 class="service-name">${service.name}</h3>
          <p class="service-description">${service.description}</p>
        </div>
      </button>
    `).join('')
    
    // Service selection
    servicesGrid.querySelectorAll('.service-card').forEach(card => {
      card.addEventListener('click', function() {
        servicesGrid.querySelectorAll('.service-card').forEach(c => {
          c.classList.remove('selected')
          c.style.borderColor = 'rgba(14, 165, 233, 0.2)'
          c.style.background = 'rgba(255, 255, 255, 0.9)'
        })
        this.classList.add('selected')
        selectedService = this.dataset.service
        const service = services.find(s => s.id === selectedService)
        if (service) {
          this.style.borderColor = service.color
          this.style.background = `linear-gradient(135deg, ${service.color}15, ${service.color}08)`
        }
        updateSubmitButton()
      })
    })
  }
  
  // Date and clinic inputs
  if (dateInput) {
    dateInput.addEventListener('change', function() {
      updateSubmitButton()
    })
  }
  
  if (clinicSelect) {
    clinicSelect.addEventListener('change', function() {
      updateSubmitButton()
    })
  }
  
  function updateSubmitButton() {
    if (submitBtn) {
      const hasDate = dateInput && dateInput.value
      const hasClinic = clinicSelect && clinicSelect.value
      submitBtn.disabled = !selectedService || !hasDate || !hasClinic
    }
  }
  
  // Form submission
  if (bookingForm) {
    bookingForm.addEventListener('submit', function(e) {
      e.preventDefault()
      
      if (!selectedService || !dateInput.value || !clinicSelect.value) {
        alert('Пожалуйста, заполните все поля')
        return
      }
      
      const service = services.find(s => s.id === selectedService)
      const newVisit = {
        id: Date.now(),
        clinic: clinicSelect.value,
        service: service.name,
        date: dateInput.value,
        status: 'active'
      }
      
      const saved = localStorage.getItem('dentalVisits')
      const visits = saved ? JSON.parse(saved) : { active: [], past: [] }
      visits.active.push(newVisit)
      localStorage.setItem('dentalVisits', JSON.stringify(visits))
      
      // Show success message
      if (successMessage) {
        successMessage.style.display = 'flex'
        setTimeout(() => {
          successMessage.style.display = 'none'
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

// Initialize slider
function initSlider() {
  const slider = document.getElementById('slider')
  if (!slider) return
  
  const slides = slider.querySelectorAll('.slide')
  const prevBtn = document.getElementById('sliderPrev')
  const nextBtn = document.getElementById('sliderNext')
  const dots = slider.querySelectorAll('.dot')
  
  let currentSlide = 0
  const totalSlides = slides.length
  
  function updateSlider() {
    slides.forEach((slide, index) => {
      slide.classList.toggle('active', index === currentSlide)
    })
    
    dots.forEach((dot, index) => {
      dot.classList.toggle('active', index === currentSlide)
    })
  }
  
  function nextSlide() {
    currentSlide = (currentSlide + 1) % totalSlides
    updateSlider()
  }
  
  function prevSlide() {
    currentSlide = (currentSlide - 1 + totalSlides) % totalSlides
    updateSlider()
  }
  
  function goToSlide(index) {
    currentSlide = index
    updateSlider()
  }
  
  if (prevBtn) prevBtn.addEventListener('click', prevSlide)
  if (nextBtn) nextBtn.addEventListener('click', nextSlide)
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => goToSlide(index))
  })
  
  // Auto-advance every 3 seconds
  setInterval(nextSlide, 3000)
  
  // Initialize
  updateSlider()
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

